<?php
/**
 * @package Tabaoca.Component.Gabble.Site
 * @subpackage com_gabble
 * @copyright (C) 2023 Jonatas C. Ferreira
 * @license GNU/AGPL v3 (see licence.txt)
 */

namespace Tabaoca\Component\Gabble\Site\Model;

\defined('_JEXEC') or die;

use stdClass;
use Joomla\CMS\MVC\Model\BaseModel;
use Joomla\CMS\Factory;
use Joomla\CMS\Session\Session;
use Joomla\CMS\Component\ComponentHelper;

/**
 * Model of Gabble Chat System component
 *
 * @package     Tabaoca.Component.Gabble.Site
 * @subpackage  com_gabble
 * @since       1.0.0
 */
class GabbleModel extends BaseModel {

	/**
	* Method to verify changed data on database.
	* 
	* @param   int  $n_users  Number of users.
	* @param   int  $n_users_on  Number of online users.
	* @param   int  $n_feed  Number of created chats.
	* @param   int  $n_notifications  Number of notifications of new messages.
	* 
	* @return  object  Response Data to XHR call.
	* @since   1.0.0
	*/
	public function run ($n_users, $n_users_on, $n_feeds, $n_notifications) { 

		$trigger = true;

		$data = new stdClass();
		$data->users_on = null;
		$data->notifications = null;
		$data->new_msgs = null;
		$data->active_id = 0;
		$data->msgs = null;

		do { 

			set_time_limit(20);

			$users_on = $this->users_on();
			$data->users_on = $users_on;

			if (($users_on->n_users_on != $n_users_on) or ($users_on->n_users != $n_users)) {

				$data->users_on = $users_on;
				$trigger = false;

			}

			$notifications = $this->feeds_notifications();

			if (($notifications->n != $n_feeds) or ($notifications->n_notifications != $n_notifications)) {

				$data->notifications = $notifications->notifications;
				$trigger = false;

			}

			$active_feed = $this->select_active_feed();

			if ($active_feed->n) {

				if ($active_feed->active[0]->active) {

					$data->active_id = $active_feed->active[0]->id;
					$new_msgs = $this->load_msgs($active_feed->active[0]->feed_id, $active_feed->active[0]->read_id, true);

					if ($new_msgs->n) {

						$this->readed_msg_feed($active_feed->active[0]->id, $new_msgs->msgs[$new_msgs->n - 1]->id);

						$data->msgs = $new_msgs->msgs;

						$trigger = false;

					}

				}

			}

			usleep(40);

		} while ($trigger);

		return $data;

	}

	/**
	* Method to return situation of opened chat windows.
	* 
	* @return  object  Response Data to XHR call.
	* @since   1.0.0
	*/
	public function situation () { 

		$app = Factory::getApplication();
		$config = $app->getParams('com_gabble');
		$cb_integration = $config->get('cb_integration');

		$currentuser = Factory::getuser();
		$usu_id = $currentuser->get("id");

		$db    = Factory::getDbo();
		$query = $db->getQuery(true);

		$query->select('a.id as id, a.feed_id as feed_id, a.user_id as user_id, a.read_id as read_id, a.w_open as w_open, a.active as active, a.w_index as w_index');
		$query->from($db->quoteName('#__gabble_notification', 'a'));
		$query->where($db->quoteName('a.user_id').'='.$usu_id);
		$query->where($db->quoteName('a.w_open').'> 0');
		$query->group('a.feed_id');
		$query->select('count(IF (b.id > a.read_id, 1, NULL)) as n_notifications');
		$query->join('LEFT', $db->quoteName('#__gabble_message', 'b') . ' ON b.feed_id = a.feed_id');
		$query->order($db->quoteName('a.w_index'));
		$query->select($db->quoteName('c.users_ids', 'users_ids'));
		$query->join('LEFT', $db->quoteName('#__gabble_feed', 'c') . ' ON c.id = a.feed_id');

		$db->setQuery($query);
		$db->execute();

		$data = new stdClass();
		$data->n = $db->getNumRows();
		$data->feeds = $db->loadObjectList();

		$data_cb_feeds = [];

		if ($data->n) {

			if ($cb_integration && ComponentHelper::isEnabled('com_comprofiler')) {

				$cb_connections = $this->select_cb_connection_list();

				for ($i = 0; $i < $data->n; $i++) {

					$data->feeds[$i]->feed_name = json_encode($this->select_feed_name(json_decode($data->feeds[$i]->users_ids, true)));

					$users_ids = json_decode($data->feeds[$i]->users_ids, false);

					for ($p = 0; $p < count($users_ids); $p++) {

						if ($users_ids[$p] != $usu_id) {

							if (in_array($users_ids[$p], $cb_connections)) {

								array_push($data_cb_feeds, $data->feeds[$i]);
								
							}

						}

					}
					
				}

				$data->feeds = $data_cb_feeds;

				for ($i = 0; $i < count($data->feeds); $i++) {

					$msgs = $this->load_msgs($data->feeds[$i]->feed_id, $data->feeds[$i]->read_id, false);
					
					if ($msgs->n) {
						
						$data->feeds[$i]->msgs = $msgs->msgs;

					} else {

						$data->feeds[$i]->msgs = null;

					}
					
				}
			
			} else {
			
				for ($i = 0; $i < $data->n; $i++) {

					$data->feeds[$i]->feed_name = json_encode($this->select_feed_name(json_decode($data->feeds[$i]->users_ids, true)));
					$msgs = $this->load_msgs($data->feeds[$i]->feed_id, $data->feeds[$i]->read_id, false);

					if ($msgs->n) {

						$data->feeds[$i]->msgs = $msgs->msgs;

					} else {

						$data->feeds[$i]->msgs = null;

					}

				}

			}

		}

		return $data;

	}

	/**
	* Method to select created feeds notifications.
	* 
	* @return  object  List of created feeds notifications.
	* @since   1.0.0
	*/
	private function feeds_notifications () {

		$app = Factory::getApplication();
		$config = $app->getParams('com_gabble');
		$cb_integration = $config->get('cb_integration');

		$currentuser = Factory::getuser();
		$usu_id = $currentuser->get("id");

		$db    = Factory::getDbo();
		$query = $db->getQuery(true);

		$query->select('a.id as id, a.feed_id as feed_id, a.user_id as user_id, a.read_id as read_id, a.w_open as w_open, a.active as active, a.w_index as w_index');
		$query->from($db->quoteName('#__gabble_notification', 'a'));
		$query->where($db->quoteName('a.user_id').'='.$usu_id);
		$query->group('a.feed_id');
		$query->select('count(IF (b.id > a.read_id, 1, NULL)) as n_notifications');
		$query->join('LEFT', $db->quoteName('#__gabble_message', 'b') . ' ON b.feed_id = a.feed_id');
		$query->select($db->quoteName('d.users_ids', 'users_ids'));
		$query->join('LEFT', $db->quoteName('#__gabble_feed', 'd') . ' ON d.id = a.feed_id');
		
		$db->setQuery($query);
		$db->execute();

		$data = new stdClass();
		$data->n = $db->getNumRows();
		$data->notifications = $db->loadObjectlist();
		$data->n_notifications = 0;

		$data_cb = [];

		if ($data->n) {

			$n_notifications = 0;
			
			if ($cb_integration && ComponentHelper::isEnabled('com_comprofiler')) {

				$cb_connections = $this->select_cb_connection_list();

				for ($i = 0; $i < $data->n; $i++) {

					
					$data->notifications[$i]->feed_name = json_encode($this->select_feed_name(json_decode($data->notifications[$i]->users_ids, true)));
					
					
					$users_ids = json_decode($data->notifications[$i]->users_ids, true);

					for ($k = 0; $k < count($users_ids); $k++) {

						if ($users_ids[$k] != $usu_id) {

							if (in_array($users_ids[$k], $cb_connections)) {

								$n_notifications = $n_notifications + $data->notifications[$i]->n_notifications;
								array_push($data_cb, $data->notifications[$i]);
								
							}

						}

					}

				}
				
				$data->n_notifications = $n_notifications;
				$data->notifications = json_decode(json_encode($data_cb), true);

			} else {

				for ($i = 0; $i < $data->n; $i++) {

					$n_notifications = $n_notifications + $data->notifications[$i]->n_notifications;
					$data->notifications[$i]->feed_name = json_encode($this->select_feed_name(json_decode($data->notifications[$i]->users_ids, true)));

				}

				$data->n_notifications = $n_notifications;

			}

		}

		return $data;

	}

	/**
	* Method to mount the name of feed with users names.
	* 
	* @param   array  $users  List of users (JSON Field).
	* 
	* @return  string  Name of feed.
	* @since   1.0.0
	*/
	private function select_feed_name ($users) {

		$app = Factory::getApplication();
		$config = $app->getParams('com_gabble');
		$cb_integration = $config->get('cb_integration');

		$data = [];
		$data_user = new stdClass;

		$currentuser = Factory::getuser();
		$usu_id = $currentuser->get("id");

		$db    = Factory::getDbo();
		$query = $db->getQuery(true);

		for ($i = 0; $i < count((array)$users); $i++) {

			if ($users[$i] != $usu_id) {

				if ($users[$i]) {

					$query->select('a.id as id, name, username');
					$query->from($db->quoteName('#__users', 'a'));
					$query->where('a.id =' . $users[$i]);

					//Community Buider Integration		
							
					if ($cb_integration && ComponentHelper::isEnabled('com_comprofiler')) {

							$query->select ('IF (cb.avatarapproved, cb.avatar, "") as avatar')
							->join('LEFT', $db->quoteName('#__comprofiler', 'cb') . ' ON cb.id = a.id');

					} else {

						$query->select ('"" as avatar');

					}

					$db->setQuery($query);
					$user = $db->loadObjectList();
					
					$data_user->user_id = $user[0]->id;
					$data_user->username = $user[0]->username;
					$data_user->avatar = $user[0]->avatar;

					array_push($data, $data_user);

				} else {

					if ($config->get('openai_gpt_name') == '') {

						$data_user->user_id = $user[0]->user_id;
						$data_user->username = 'Gabble Bot';
						$data_user->avatar = '';
						
						array_push($data, $data_user);

					} else {
						
						$data_user->user_id = $user[0]->user_id;
						$data_user->username = 'Gabble Bot';
						$data_user->avatar = $config->get('openai_gpt_name');
						array_push($data, $data_user);

					}

				}

			}

		}

		return $data;

	}

	/**
	* Method to select messages to populate chat window.
	* 
	* @param   int  $feed_id  Id of Feed.
	* @param   int  $read_id  Id of the last message readed in feed.
	* @param   bool  $new_msgs  True for only messages after last message readed. False to messages before.
	* 
	* @return  object  List of messages.
	* @since   1.0.0
	*/
	private function load_msgs ($feed_id, $read_id, $new_msgs) { 

		$currentuser = Factory::getuser();
		$usu_id = $currentuser->get("id");

		$db = Factory::getDbo();
		$query = $db->getQuery(true);

		$query->select('a.id as id, a.feed_id as feed_id, a.user_id as user_id, a.message as message, a.sended as sended, a.react as react, a.answer as answer, a.files as files, a.params as params');
		$query->from($db->quoteName('#__gabble_message', 'a'));
		$query->where('feed_id =' . $feed_id);
		$query->select($db->quoteName('u.username', 'username'));
		$query->join('LEFT', $db->quoteName('#__users', 'u') . ' ON u.id = a.user_id');

		if ($new_msgs) {

			$query->where('a.id >' . $read_id);

		} else {

			$query->where('a.id <=' . $read_id);

		}

		//Community Buider Integration		
				
		$app = Factory::getApplication();

		$config = $app->getParams('com_gabble');
		$cb_integration = $config->get('cb_integration');

		if ($cb_integration) {

			if (ComponentHelper::isEnabled('com_comprofiler')) {

				$query->select ('IF (cb.avatarapproved, cb.avatar, "") as avatar')
				->join('LEFT', $db->quoteName('#__comprofiler', 'cb') . ' ON cb.id = a.user_id');

			} else {

				$query->select ('"" as avatar');

			}

		} else {

			$query->select ('"" as avatar');

		}

		$query->order('a.id');

		$db->setQuery($query);
		$db->execute();

		$data = new stdClass();
		$data->n = $db->getNumRows();
		$data->msgs = $db->loadObjectList();

		if ($data->n) {

			for ($i = 0; $i < $data->n; $i++) {

				$data->msgs[$i]->message = $this->decrypt_message($data->msgs[$i]->message, $data->msgs[$i]->feed_id);

			}

		}

		return $data;

	}

	/**
	* Method to select the active feed.
	* 
	* @return  object  Active feed.
	* @since   1.0.0
	*/
	private function select_active_feed () {

		$currentuser = Factory::getuser();
		$usu_id = $currentuser->get("id");

		$db = Factory::getDbo();
		$query = $db->getQuery(true);

		$query->select('a.id as id, a.feed_id as feed_id, a.user_id as user_id, a.read_id as read_id, a.w_open as w_open, a.active as active, a.w_index as w_index');
		$query->from($db->quoteName('#__gabble_notification', 'a'));
		$query->where($db->quoteName('a.user_id').'='.$usu_id);
		$query->where($db->quoteName('a.w_open').'> 0');
		$query->where($db->quoteName('a.active').'= 1');

		$db->setQuery($query);
		$db->execute();

		$data = new stdClass();
		$data->n = $db->getNumRows();
		$data->active = $db->loadObjectList();

		return $data;

	}
	
	/**
	* Method to Community Builder Integration.
	* 
	* @param   array  $users  List of users (JSON Field).
	* 
	* @return  string  Name of feed.
	* @since   1.0.0
	*/
	private function select_cb_connection () {

		$app = Factory::getApplication();
		$config = $app->getParams('com_gabble');
		$currentuser = Factory::getuser();
		$user_id = $currentuser->get("id");

		$data = new stdClass;

		$db    = Factory::getDbo();
		$query = $db->getQuery(true);
		
		$query->select('a.referenceid as referenceid, a.memberid as memberid, a.accepted as accepted, a.pending as pending, a.membersince as membersince, a.reason as reason, a.description as description, a.type as type');
		$query->from($db->quoteName('#__comprofiler_members', 'a'));
		$query->where($db->quoteName('a.referenceid').'='.$user_id);
		$query->where($db->quoteName('a.accepted').'= 1');
		$query->where($db->quoteName('a.pending').'= 0');
		$query->select ('IF (cb.avatarapproved, cb.avatar, "") as avatar')
			->join('LEFT', $db->quoteName('#__comprofiler', 'cb') . ' ON cb.id = a.memberid');

		
		$db->setQuery($query);
		$db->execute();
		$data->n_cb_users = $db->getNumRows();
		$data->cb_users = $db->loadObjectlist();

		return $data;

	}

	/**
	* Method to Community Builder Integration.
	* 
	* @param   array  $users  List of users (JSON Field).
	* 
	* @return  string  Name of feed.
	* @since   1.0.0
	*/
	private function select_cb_connection_list () {

		$data_users = $this->select_cb_connection();
		$data = [];

		if ($data_users->n_cb_users) {

			for ($i = 0; $i < $data_users->n_cb_users; $i++) {
			
				array_push($data, $data_users->cb_users[$i]->memberid);

			}

		}

		return $data;

	}

	/**
	* Method to select online users.
	* 
	* @return  object  All Users List.
	* @since   1.0.0
	*/
    private function users_on () {

		$currentuser = Factory::getuser();
		$usu_id = $currentuser->get("id");

		$db = Factory::getDbo();
		$query_a = $db->getQuery(true);

		$query_a->select('a.id as id, a.name as name, a.username as username');
		$query_a->from($db->quoteName('#__users', 'a'));
		$query_a->where('a.id !=' . $usu_id);

		//Community Builder Integration

		$app = Factory::getApplication();

		$config = $app->getParams('com_gabble');
		$cb_integration = $config->get('cb_integration');

		if ($cb_integration) {

			if (ComponentHelper::isEnabled('com_comprofiler')) {

				$cb_connections = $this->select_cb_connection_list();

				$query_a->where('a.id IN (' . implode(',', $cb_connections) . ')');
				
				$query_a->select ('IF (cb.avatarapproved, cb.avatar, "") as avatar')
				->join('LEFT', $db->quoteName('#__comprofiler', 'cb') . ' ON cb.id = a.id');

			} else {

				$query_a->select ('"" as avatar');

			}

		} else {

			$query_a->select ('"" as avatar');

		}
		
		$db->setQuery($query_a);
		$db->execute();
		$n_users = $db->getNumRows();
		$users = $db->loadObjectList();

		$db = Factory::getDbo();
		$query_b = $db->getQuery(true);

		$query_b->select($db->quoteName(array('userid','username')));
		$query_b->from($db->quoteName('#__session'));
		$query_b->where('guest = 0 AND client_id = 0 AND userid != ' . $usu_id);

		$db->setQuery($query_b);
		$db->execute();
		$n_users_on = $db->getNumRows();
		$users_on = $db->loadObjectlist();

		for ($i = 0; $i < $n_users; $i++) {

			for($t = 0; $t < $n_users_on; $t++) {

				if ($users[$i]->id == $users_on[$t]->userid) {

					$users[$i]->online = 1;

				} else {

					$users[$i]->online = 0;

				}

			}

		}

		$data = new stdClass();
		$data->users = $users;
		$data->n_users = $n_users;
		$data->n_users_on = $n_users_on;

		return $data;

	}

	/**
	* Method to select notification data for a feed by user id.
	* 
	* @param   int  $user_id  Id of User.
	* @param   int  $feed_id  Id of Feed.
	* 
	* @return  object  Notification.
	* @since   1.0.0
	*/
	private function select_notification_by_user ($user_id, $feed_id) {

		$db    = Factory::getDbo();
		$query = $db->getQuery(true);

		$query->select('a.id as id, a.feed_id as feed_id, a.user_id as user_id, a.read_id as read_id, a.w_open as w_open, a.active as active, a.w_index as w_index');
		$query->from($db->quoteName('#__gabble_notification', 'a'));
		$query->where($db->quoteName('a.user_id').'='.$user_id);
		$query->where($db->quoteName('a.feed_id').'='.$feed_id);
		$query->group('a.feed_id');
		$query->select($db->quoteName('b.users_ids', 'users_ids'));
		$query->join('LEFT', $db->quoteName('#__gabble_feed', 'b') . ' ON b.id = a.feed_id');

		$db->setQuery($query);
		$db->execute();
		$data = new stdClass();
		$data->n = $db->getNumRows();
		$data->notification = $db->loadObjectList();

		if ($data->n) {

			for ($i = 0; $i < $data->n; $i++) {

				$data->notification[$i]->feed_name = json_encode($this->select_feed_name(json_decode($data->notification[$i]->users_ids, true)));

			}

		}

		return $data;

	}

	/**
	* Method to select notification data for a feed by notification id.
	* 
	* @param   int  $notification_id  Id of Notification for a feed.
	* 
	* @return  object  Notification.
	* @since   1.0.0
	*/
	private function select_notification_by_id ($notification_id) {

		$db    = Factory::getDbo();
		$query = $db->getQuery(true);

		$query->select('a.id as id, a.feed_id as feed_id, a.user_id as user_id, a.read_id as read_id, a.w_open as w_open, a.active as active, a.w_index as w_index');
		$query->from($db->quoteName('#__gabble_notification', 'a'));
		$query->where($db->quoteName('a.id').'='.$notification_id);
		$query->group('a.feed_id');
		$query->select($db->quoteName('b.users_ids', 'users_ids'));
		$query->join('LEFT', $db->quoteName('#__gabble_feed', 'b') . ' ON b.id = a.feed_id');

		$db->setQuery($query);
		$db->execute();		
		$data = new stdClass();
		$data->n = $db->getNumRows();
		$data->notification = $db->loadObjectList();

		if ($data->n) {

			for ($i = 0; $i < $data->n; $i++) {

				$data->notification[$i]->feed_name = json_encode($this->select_feed_name(json_decode($data->notification[$i]->users_ids, true)));

			}

		}

		return $data;

	}

	/**
	* Method to select feed id by user id to talk.
	* 
	* @param   int  $talkto_id  Id of User to talk.
	* 
	* @return  int  Id of Feed.
	* @since   1.0.0
	*/
	private function select_feed_id ($talkto_id) { 

		$currentuser = Factory::getuser();
		$usu_id = $currentuser->get("id");

		$db = Factory::getDbo();
		$query = $db->getQuery(true);

		$query->select($db->quoteName(array('id','users_ids')))
				->from($db->quoteName('#__gabble_feed'))
				->where('JSON_LENGTH(users_ids) = 2 AND JSON_CONTAINS(users_ids, '. $db->quote($usu_id) . ' ) AND JSON_CONTAINS(users_ids, '. $db->quote($talkto_id) . ' )');

		$db->setQuery($query);
		$db->execute();
		$n = $db->getNumRows();
		$feed = $db->loadObjectlist();

		if ($n) {

			return $feed[0]->id;

		} else {

			return 0;

		}

	}

	/**
	* Method to create a new feed record on database by user id to talk.
	* 
	* @param   int  $talkto_id  Id of User to talk.
	* 
	* @return  void
	* @since   1.0.0
	*/
	private function create_feed ($talkto_id) {

		$currentuser = Factory::getuser();
		$usu_id = $currentuser->get("id");

		$db = Factory::getDbo();
		
		$users= array($usu_id, $talkto_id);

		$feed = new stdClass();

		$feed->users_ids = json_encode($users);
		$feed->type = 0;
		$feed->params = json_encode([]);

		$result = $db->insertObject('#__gabble_feed', $feed);

		$notification = new stdClass();
		$notification->feed_id = $this->select_feed_id($talkto_id);
		$notification->user_id = $usu_id;
		$notification->read_id = 0;
		$notification->open = 0;
		$notification->active = 0;
		$notification->w_index = 0;
		$notification->date_in = date('Y-m-d H:i:s');
		$notification->date_out = date('Y-m-d H:i:s');
		$notification->params = json_encode([]);

		$result_a = $db->insertObject('#__gabble_notification', $notification);

		$notification->user_id = $talkto_id;

		$result_b = $db->insertObject('#__gabble_notification', $notification);

	}

	/**
	* Method to open chat window by user id to talk.
	* 
	* @param   int  $talkto_id  Id of User to talk.
	* 
	* @return  object  Response Data to XHR call.
	* @since   1.0.0
	*/
	public function open_user ($talkto_id) {

		$currentuser = Factory::getuser();
		$usu_id = $currentuser->get("id");

		$feed_id = $this->select_feed_id($talkto_id);

		if ($feed_id > 0) {

			$data = $this->select_notification_by_user($usu_id, $feed_id);
			$data->open = false;

			if ($data->n) {

				if ($data->notification[0]->w_open) {

					$data->open = true;
					$this->activate_feed($data->notification[0]->id);

				} else {

					$this->activate_feed($data->notification[0]->id);
					$msgs = $this->load_msgs($data->notification[0]->feed_id, $data->notification[0]->read_id, false);

					if ($msgs->n) {

						$data->notification[0]->msgs = $msgs->msgs;

					} else {

						$data->notification[0]->msgs = null;

					}

				}

			}

		} else {

			$this->create_feed($talkto_id);

			$feed_id_created = $this->select_feed_id($talkto_id);

			$data = $this->select_notification_by_user($usu_id, $feed_id_created);
			$data->open = false;

			if ($data->n) {

				$this->activate_feed($data->notification[0]->id);

			}

		}

		return $data;

	}

	/**
	* Method to open chat window.
	* 
	* @param   int  $notification_id  Id of Notification.
	* 
	* @return  object  Response Data to XHR call.
	* @since   1.0.0
	*/
	public function open_feed ($notification_id) {

		$this->activate_feed($notification_id);

		$data = $this->select_notification_by_id($notification_id);

		if ($data->n) {

			$msgs = $this->load_msgs($data->notification[0]->feed_id, $data->notification[0]->read_id, false);

			if ($msgs->n) {

				$data->notification[0]->msgs = $msgs->msgs;

			} else {

				$data->notification[0]->msgs = null;

			}

			$data->notification[0]->feed_name = json_encode($this->select_feed_name(json_decode($data->notification[0]->users_ids, true)));

		}

		return $data;

	}

	/**
	* Method to close chat window.
	* 
	* @param   int  $notification_id  Id of Notification.
	* 
	* @return  object  Response Data to XHR call.
	* @since   1.0.0
	*/
	public function close_feed ($notification_id) {

		$currentuser = Factory::getuser();
		$usu_id = $currentuser->get("id");

		$data = $this->select_notification_by_id($notification_id);

		$db    = Factory::getDbo();
		$query = $db->getQuery(true);

		$query->select('a.id as id, a.feed_id as feed_id, a.user_id as user_id, a.read_id as read_id, a.w_open as w_open, a.active as active, a.w_index as w_index');
		$query->from($db->quoteName('#__gabble_notification', 'a'));
		$query->where($db->quoteName('a.user_id').'='.$usu_id);
		$query->where($db->quoteName('a.w_index').'>'.$data->notification[0]->w_index);

		$db->setQuery($query);
		$db->execute();

		$n = $db->getNumRows();
		$notifications = $db->loadObjectList();

		$a = new stdClass();
		$a->id = $notification_id;
		$a->w_open = 0;
		$a->active = 0;
		$a->w_index = 0;

		$up = $db->updateObject('#__gabble_notification', $a, 'id');

		if ($n) {

			for ($i = 0; $i < $n; $i++) {

				$b = new stdClass();
				$b->id = $notifications[$i]->id;
				$b->w_index = $notifications[$i]->w_index - 1;

				$up = $db->updateObject('#__gabble_notification', $b, 'id');

			}

		}

		return $data;

	}

	/**
	* Method to minimize chat window.
	* 
	* @param   int  $notification_id  Id of Notification.
	* 
	* @return  object  Response Data to XHR call.
	* @since   1.0.0
	*/
	public function minimize_feed ($notification_id) {

		$currentuser = Factory::getuser();
		$usu_id = $currentuser->get("id");

		$data = $this->select_notification_by_id($notification_id);

		$a = new stdClass();
		$a->id = $notification_id;

		if ($data->n) {

			if ($data->notification[0]->w_open == 2) {

				$a->w_open = 1;

				$active = $this->select_active_feed();

				if ($active->n) {

					$this->activate_feed($notification_id);

				} else {

					$a->active = 1;

				}

			} else {

				$a->w_open = 2;
				$a->active = 0;

			}

		}

		$db = Factory::getDbo();
		$up = $db->updateObject('#__gabble_notification', $a, 'id');

		return $a;

	}

	/**
	* Method to activate chat window.
	* 
	* @param   int  $notification_id  Id of Notification.
	* 
	* @return  object  Response Data to XHR call.
	* @since   1.0.0
	*/
	public function activate_feed ($notification_id) {

		$active_feed = $this->select_active_feed();

		$data = $this->select_notification_by_id($notification_id);

		$feeds_open = $this->situation();

		if ($active_feed->n) {

			if ($active_feed->active[0]->id != $notification_id) {

				$b = new stdClass();
				$b->id = $active_feed->active[0]->id;
				$b->active = 0;

				$db = Factory::getDbo();

				$up = $db->updateObject('#__gabble_notification', $b, 'id');

				$a = new stdClass();
				$a->id = $notification_id;
				$a->active = 1;

				if ($data->notification[0]->w_open != 1) {

					$a->w_open = 1;
					$a->w_index = $feeds_open->n + 1;

				} 

				$db = Factory::getDbo();

				$up = $db->updateObject('#__gabble_notification', $a, 'id');

			}

		} else {

			$a = new stdClass();
			$a->id = $notification_id;
			$a->active = 1;

			if ($data->notification[0]->w_open != 1) {

				$a->w_open = 1;
				$a->w_index = $feeds_open->n + 1;

			} 

			$db = Factory::getDbo();

			$up = $db->updateObject('#__gabble_notification', $a, 'id');

		}

		return $data;

	}

	/**
	* Method to update last message readed.
	* 
	* @param   int  $notification_id  Id of Notification.
	* @param   int  $read_id  Id of last message readed.
	* 
	* @return  object  Notification data of active Feed.
	* @since   1.0.0
	*/
	private function readed_msg_feed($notification_id, $read_id) {

		$a = new stdClass();
		$a->id = $notification_id;
		$a->read_id = $read_id;

		$db = Factory::getDbo();

		$up = $db->updateObject('#__gabble_notification', $a, 'id');

		$data = $this->select_active_feed();

		return $data;

	}

	/**
	* Method to send message to Feed.
	* 
	* @param   int  $notification_id  Id of Notification.
	* @param   int  $feed_id  Id of Feed.
	* @param   string  $msg  Message to send.
	* @param   bool  $bot  True for message from OpenAI GPT Bot.
	* 
	* @return  object  Response Data to XHR call.
	* @since   1.0.0
	*/
	public function send_msg_feed ($notification_id, $feed_id, $msg, $bot) {

		$currentuser = Factory::getuser();
		$usu_id = $currentuser->get("id");

		$msg = rawurldecode($msg);

		$data = new stdClass();

		$msg = str_replace("<", "&lt", $msg);
		$msg = str_replace(">", "&gt", $msg);

		if ($bot) {

			$usu_id = 0;
			$msg = str_replace("\n", "<br>", $msg);

			$p = explode("```", $msg);

			if (count($p) > 2) {

				for ($i = 1; $i < count($p); $i++) {

					$t = explode("```", $msg, 2);

					if (!is_integer($i/2)) {

						$msg = implode("<code>", $t);

					} else {

						$msg = implode("</code>", $t);

					}

				}

			}

		}

		$message = new stdClass();

		$message->user_id = $usu_id;
		$message->feed_id = $feed_id;
		$message->message = $this->encrypt_message($msg, $feed_id);
		$message->sended = date('Y-m-d H:i:s');
		$message->react = json_encode([]);
		$message->answer = 0;
		$message->files = json_encode([]);
		$message->params = json_encode([]);

		$db = Factory::getDbo();
		$resultmsg = $db->insertObject('#__gabble_message', $message);

		$data->notification_id = $notification_id;

		return $data;

	}

	/**
	* Method to send message to OpenAI GPT.
	* 
	* @param   int  $notification_id  Id of Notification.
	* @param   int  $feed_id  Id of Feed.
	* @param   string  $msg  Message to send.
	* 
	* @return  object  Response Data to XHR call.
	* @since   1.0.0
	*/
	public function send_msg_tobot ($notification_id, $feed_id, $msg) {

		$app = Factory::getApplication();
		$config = $app->getParams('com_gabble');

		$data = new stdClass();
		$data->notification_id = $notification_id;

		if ($config->get('openai_gpt')) {

			$last_msgs = $this->last_gpt_msgs($feed_id);

			$this->send_msg_feed($notification_id, $feed_id, $msg, 0);

			$msg = rawurldecode($msg);

			$response = $this->openai_get_message($msg, $last_msgs);

			if ($response->bot_error == false) {

				$object = json_decode($response->body, true);

				$msg_bot = $object['choices'][0];
				$data = $this->send_msg_feed($notification_id, $feed_id, $msg_bot['message']['content'], 1);

				$data->body = $object;
				$data->response = $response;

			}

			$data->bot_error = $response->bot_error;

		} else {
			
			$data->bot_error = true;
			
		}

		return $data;

	}

	/**
	* Method to get message from OpenAI GPT using cURL.
	* 
	* @param   string  $message  Message to send.
	* @param   string  $last_msgs  Last messages sended.
	* 
	* @return  object  Response Data from OpenAI GPT.
	* @since   1.0.0
	*/
	private function openai_get_message($message, $last_msgs) {

		$currentuser = Factory::getuser();
		$usu_id = $currentuser->get("id");

		$app = Factory::getApplication();
		$config = $app->getParams('com_gabble');

		$openai_url = $config->get('openai_gpt_link');

		$headers = array( 'Content-Type: application/json', 'Authorization: Bearer '. $config->get('openai_api_key'));

		$data = new stdClass();

		if ($last_msgs->user_msg != null) {

			$data->messages = array(
								array("role" => "user", "content" => $last_msgs->user_msg), 
								array("role" => "assistant", "content" => $last_msgs->gpt_msg),
								array("role" => "user", "content" => $message));

		} else {

			$data->messages = array(array("role" => "user", "content" => $message));

		}

		$data->model = $config->get('openai_gpt_model');

		$data->user = strval($usu_id);

		if (intval($config->get('openai_gpt_max_tokens'))) {

			$data->max_tokens = intval($config->get('openai_gpt_max_tokens'));

		}

		if (floatval($config->get('openai_gpt_temperature'))) {

			$data->temperature = floatval($config->get('openai_gpt_temperature'));

		}

		if (floatval($config->get('openai_gpt_presence_penalty'))) {

			$data->presence_penalty = floatval($config->get('openai_gpt_presence_penalty'));

		}

		if (floatval($config->get('openai_gpt_frequency_penalty'))) {

			$data->frequency_penalty = floatval($config->get('openai_gpt_frequency_penalty'));

		}

		$gpt_fields = json_encode($data);

		$ch = curl_init();

		curl_setopt($ch, CURLOPT_URL, $openai_url);

		curl_setopt($ch, CURLOPT_POST, true);
		curl_setopt($ch, CURLOPT_HEADER, true);
		curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
		curl_setopt($ch, CURLOPT_POSTFIELDS, $gpt_fields);
		curl_setopt($ch, CURLOPT_FAILONERROR, true);
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
		curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 5);

		$response = curl_exec($ch);

		$result = new stdClass();

		if (curl_error($ch) != '') {

			$bot_error = true;
			$result->bot_error = $bot_error;

		} else {

			$bot_error = false;
			$info = curl_getinfo($ch);
			$result->header = substr($response, 0, $info["header_size"]);
			$result->body = substr($response, $info["header_size"]);
			$result->bot_error = $bot_error;
			$result->gpt_fields = $gpt_fields;

		}

		curl_close($ch);

		return $result;

	}

	/**
	* Method to get last messages from OpenAI GPT to chat.
	* 
	* @param   int  $feed_id  Id of Feed.
	* 
	* @return  object  Messages to send.
	* @since   1.0.0
	*/
	private function last_gpt_msgs ($feed_id) {

		$currentuser = Factory::getuser();
		$usu_id = $currentuser->get("id");

		$data = new stdClass();
		$data->user_msg = null;
		$data->gpt_msg = null;

		$db = Factory::getDbo();
		$query_user = $db->getQuery(true);

		$query_user->select('id, feed_id, user_id, message');
		$query_user->from($db->quoteName('#__gabble_message'));
		$query_user->where($db->quoteName('user_id') . '=' . $usu_id);
		$query_user->where($db->quoteName('feed_id') . '=' . $feed_id);
		$query_user->order('id DESC');

		$db->setQuery($query_user);
		$db->execute();

		$u = $db->getNumRows();
		$user_msg = $db->loadObjectList();

		$db = Factory::getDbo();
		$query_gpt = $db->getQuery(true);

		$query_gpt->select('id, feed_id, user_id, message');
		$query_gpt->from($db->quoteName('#__gabble_message'));
		$query_gpt->where($db->quoteName('user_id') . '= 0');
		$query_gpt->where($db->quoteName('feed_id') . '=' . $feed_id);
		$query_gpt->order('id DESC');

		$db->setQuery($query_gpt);
		$db->execute();

		$g = $db->getNumRows();
		$gpt_msg = $db->loadObjectList();

		if ($u) {

			if ($g) {

				if ($user_msg[0]->id < $gpt_msg[0]->id) {

					$data->user_msg = $this->decrypt_message($user_msg[0]->message, $feed_id);
					$data->gpt_msg = $this->decrypt_message($gpt_msg[0]->message, $feed_id);

				} else {

					$data->user_msg = null;
					$data->gpt_msg = null;

				}

			}

		}

		return $data;

	}

	/**
	* Method to encrypt message.
	* 
	* @param   string  $message  Messsage to encrypt.
	* @param   int  $key  Key to encrypt.
	* 
	* @return  string  Message encryped.
	* @since   1.0.0
	*/
	private function encrypt_message ($message, $key) {

		$algoritmo = "AES-256-CTR";
		$key = $key."gabble";
		$iv = "UuUfdNuUv5WYqF65";
		$tag = "";
		$tag_length = 16;

		$message = openssl_encrypt($message, $algoritmo, $key, OPENSSL_RAW_DATA, $iv, $tag, "", $tag_length);
		$message = base64_encode($message);

		return $message;

	}

	/**
	* Method to decrypt message.
	* 
	* @param   string  $message  Messsage to decrypt.
	* @param   int  $key  Key to decrypt.
	* 
	* @return  string  Message decryped.
	* @since   1.0.0
	*/
	private function decrypt_message ($message, $key) {

		$algoritimo = "AES-256-CTR";
		$key = $key."gabble";
		$iv = "UuUfdNuUv5WYqF65";

		$message = openssl_decrypt(base64_decode($message), $algoritimo, $key, OPENSSL_RAW_DATA, $iv);

		return $message;

	}
	

}
