<?php
/**
 * @package Tabaoca.Component.Cotton.Site
 * @subpackage com_cotton
 * @copyright (C) 2024 Jonatas C. Ferreira
 * @license GNU/AGPL v3 https://www.gnu.org/licenses/agpl-3.0.html
 */

namespace Tabaoca\Component\Cotton\Site\Model;

\defined('_JEXEC') or die;

use stdClass;
use Joomla\CMS\MVC\Model\BaseModel;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Uri\Uri;

/**
 * Model of Cotton Cloud System component
 *
 * @package     Tabaoca.Component.Cotton.Site
 * @subpackage  com_cotton
 * @since       1.0.0
 */
class CottonModel extends BaseModel {

	/**
	* Constant to check max file size to database LONGBLOB field.
	*/ 
	protected $max_filesize_db = 4294967295;
	
	/**
	* Method to get inicial data information to start the Cotton Cloud System.
	* 
	* @return  object  Root folder content information and configuration data of system.
	* @since   1.0.0
	*/
	public function start () {

		$currentuser = Factory::getApplication()->getIdentity();
		$user_id = $currentuser->get("id");

		$app = Factory::getApplication();
		$config = $app->getParams('com_cotton');

		$db = $this->getDatabase();
		$query = $db->getQuery(true);

		$query->select('a.id as id');
		$query->from($db->quoteName('#__cotton_folder', 'a'));
		$query->where($db->quoteName('a.owner_id').'='.$user_id);
		$query->where($db->quoteName('a.parent_id').'= 0');
		$query->where($db->quoteName('a.trash').'= 0');

		$db->setQuery($query);
		$db->execute();

		$n = $db->getNumRows();
		$parent = $db->loadObjectList();

		if ($n) {

			$data = $this->folder_load($parent[0]->id);
			$data->config = new stdClass();
			$data->config->text_formats = $this->array_formats($config->get('cotton_text_formats', 'txt,md,ini,htm,html,xhtml,xml,js,mjs,php,css,sass,scss,less,csv,ics,json,jsonld,xul'));
			$data->config->image_formats = $this->array_formats($config->get('cotton_image_formats', 'bmp,jpg,jpeg,jfif,pjpeg,pjp,png,apng,gif,svg,webp'));
			$data->config->video_formats = $this->array_formats($config->get('cotton_video_formats', 'mp4,webm,ogg'));
			$data->config->audio_formats = $this->array_formats($config->get('cotton_audio_formats', 'wav,mp3,ogg'));
			$data->config->config_header = $config->get('cotton_header');

		} else {

			//Create root folder
			$archive = $this->folder_create(0, 'Archive', '');
			//Reload
			$data = $this->start();

		}

		return $data;

	}

	/**
	* Method to load all items information into of a folder.
	*
	* @param   int  $folder_id Folder id to load data.
	* 
	* @return  object  Folder content data (folders, files, root folder tree and limits).
	* @since   1.0.0
	*/
	public function folder_load ($folder_id) {

		$currentuser = Factory::getApplication()->getIdentity();
		$user_id = $currentuser->get("id");
		
		$data = new stdClass();
		
		$db = $this->getDatabase();
		$query = $db->getQuery(true);

		$query->select('a.id as id, a.owner_id as owner_id, a.name as name, a.description as description, a.date_created as date_created, a.date_updated as date_updated, 
						a.featured as featured, a.parent_id as parent_id, a.allowed_users as allowed_users, a.open_link as open_link, a.trash as trash, a.params as params');
		$query->from($db->quoteName('#__cotton_folder', 'a'));
		$query->where($db->quoteName('a.parent_id').'='.$folder_id);
		$query->where($db->quoteName('a.owner_id').'='.$user_id);
		$query->where($db->quoteName('a.trash').'= 0');
		$query->group('a.id');
		$query->select('COUNT(DISTINCT b.id) as n_folders');
		$query->join('LEFT', $db->quoteName('#__cotton_folder', 'b') . ' ON b.parent_id = a.id AND b.owner_id = ' . $user_id . ' AND b.trash = 0');
		$query->select('COUNT(DISTINCT c.id) as n_files');
		$query->join('LEFT', $db->quoteName('#__cotton_file', 'c') . ' ON c.folder_id = a.id AND c.owner_id = ' . $user_id . ' AND c.trash = 0');
		$query->order($db->quoteName('name'));

		$db->setQuery($query);
		$db->execute();

		$data->n_folders = $db->getNumRows();
		$data->folders = $db->loadObjectList();

		$files = $this->folder_load_files($folder_id);

		$data->n_files = $files->n;
		$data->files = $files->files;
		
		$folder_list = $this->folder_load_list();

		$data->n_list = $folder_list->n;
		$data->list = $folder_list->list;

		$data->active_folder_id = $folder_id;

		$data->limits = $this->limit_file_space();

		return $data;

	}

	/**
	* Method to load path tree folder.
	*
	* @param   int  $folder_id Folder id to create path.
	* 
	* @return  string  Path from a folder.
	* @since   1.0.0
	*/
	private function folder_load_path ($folder_id) {

		$currentuser = Factory::getApplication()->getIdentity();
		$user_id = $currentuser->get("id");
		
		$data = new stdClass();
		
		$db = $this->getDatabase();
		$query = $db->getQuery(true);

		$query->select('a.id as id, a.name as name, a.parent_id as parent_id, a.open_link as open_link');
		$query->from($db->quoteName('#__cotton_folder', 'a'));
		$query->where($db->quoteName('a.id').'='.$folder_id);
		$query->where($db->quoteName('a.owner_id').'='.$user_id);
		$query->where($db->quoteName('a.trash').'= 0');
		$query->order($db->quoteName('name'));

		$db->setQuery($query);
		$db->execute();
		$n = $db->getNumRows();

		$folder = $db->loadObjectList();

		if ($n) {

			$path = $this->folder_load_path($folder[0]->parent_id) . '/' . $folder[0]->name;

		} else {

			$path = '';

		}

		return $path;

	}

	/**
	* Method to load the root folder tree list.
	* 
	* @return  object  List of folders with your parent_id.
	* @since   1.0.0
	*/
	private function folder_load_list () {

		$currentuser = Factory::getApplication()->getIdentity();
		$user_id = $currentuser->get("id");
		
		$data = new stdClass();
		
		$db = $this->getDatabase();
		$query = $db->getQuery(true);

		$query->select('a.id as id, a.name as name, a.parent_id as parent_id, a.open_link as open_link');
		$query->from($db->quoteName('#__cotton_folder', 'a'));
		$query->where($db->quoteName('a.owner_id').'='.$user_id);
		$query->where($db->quoteName('a.trash').'= 0');
		$query->order($db->quoteName('name'));

		$db->setQuery($query);
		$db->execute();
		$data->n = $db->getNumRows();

		$data->list = $db->loadObjectList();

		if ($data->n) {

			for ($i = 0; $i < $data->n; $i++) {

				$data->list[$i]->path = $this->folder_load_path($data->list[$i]->id);

			}

		}

		return $data;

	}

	/**
	* Method to create a new folder item in the database.
	* 
	* @param   int  $parent_id  Id of parent folder.
	* @param   string  $folder_name  The name to new folder.
	* @param   string  $folder_description The description to new folder.
	* 
	* @return  object  status and $parent_id
	* @since   1.0.0
	*/
	public function folder_create ($parent_id, $folder_name, $folder_description) {

		$currentuser = Factory::getApplication()->getIdentity();
		$user_id = $currentuser->get("id");

		$db = $this->getDatabase();

		$folder = new stdClass();

		$folder->name = $folder_name;
		$folder->description = $folder_description;
		$folder->date_created = date('Y-m-d H:i:s');
		$folder->date_updated = date('Y-m-d H:i:s');
		$folder->parent_id = $parent_id;
		$folder->owner_id = $user_id;
		$folder->allowed_users = json_encode([]);
		$folder->open_link = 0;
		$folder->trash = 0;
		$folder->params = json_encode([]);

		$result = $db->insertObject('#__cotton_folder', $folder);

		$data = new stdClass();
		$data->success = true;
		$data->error = '';
		$data->parent_id = $parent_id;

		return $data;

	}

	/**
	* Method to load all files information from a folder content.
	*
	* @param   int  $folder_id Folder id to load files.
	* 
	* @return  object  List of files information.
	* @since   1.0.0
	*/
	private function folder_load_files ($folder_id) {

		$currentuser = Factory::getApplication()->getIdentity();
		$user_id = $currentuser->get("id");

		$db = $this->getDatabase();
		$query = $db->getQuery(true);

		$query->select('a.id as id, a.owner_id as owner_id, a.name as name, a.description as description, a.date_created as date_created, a.date_updated as date_updated, 
						a.featured as featured, a.downloads as downloads, a.size as size, a.mime_type as mime_type, a.folder_id as folder_id, a.allowed_users as allowed_users, a.open_link as open_link, a.trash as trash, a.params as params');
		$query->from($db->quoteName('#__cotton_file', 'a'));
		$query->where($db->quoteName('a.folder_id').'='.$folder_id);
		$query->where($db->quoteName('a.owner_id').'='.$user_id);
		$query->where($db->quoteName('a.trash').'= 0');
		$query->order($db->quoteName('name'));

		$db->setQuery($query);
		$db->execute();

		$data = new stdClass();
		$data->n = $db->getNumRows();
		$data->files = $db->loadObjectList();

		if ($data->n) {

			for ($i = 0; $i < $data->n; $i++) {

				$data->files[$i]->path = $this->folder_load_path($data->files[$i]->folder_id);

			}

		}

		return $data;

	}

	/**
	* Method to update data information from a folder item.
	* 
	* @param   int  $folder_id  Id of folder to be updated.
	* @param   string  $folder_name  The name to folder.
	* @param   string  $folder_description The description to folder.
	* 
	* @return  object  Folder data updated.
	* @since   1.0.0
	*/
	public function folder_update ($folder_id, $folder_name, $folder_description) {

		$folder = new stdClass();
		$folder->id = $folder_id;
		$folder->name = $folder_name;
		$folder->date_updated = date('Y-m-d H:i:s');
		$folder->description = $folder_description;

		$db = $this->getDatabase();

		$up = $db->updateObject('#__cotton_folder', $folder, 'id');

		return $folder;

	}

	/**
	* Method to delete a folder item and all your content.
	* 
	* @param   int  $folder_id  Id of folder to be deleted.
	* @param   bool  $trash  If send to trash or delete from database.
	* 
	* @return  void
	* @since   1.0.0
	*/
	public function folder_delete ($folder_id, $trash) {

		$recursive = $this->folder_load($folder_id);

		if ($trash == 2) {

			$recover = true;

		} else {

			$recover = false;

		}

		if ($recursive->n_folders) {
		
			for ($p = 0; $p < $recursive->n_folders; $p++) {
			
				$this->folder_delete($recursive->folders[$p]->id, $trash); //recursive

			}
			
			if ($trash) {
				
				$this->folder_items_trash($recursive->folders, 'folder', $recover);

			} else {

				$this->folder_items_delete($recursive->folders, 'folder');

			}

		}

		if ($recursive->n_files) {
		
			if ($trash) {
				
				$this->folder_items_trash($recursive->files, 'file', $recover);

			} else {

				$this->folder_items_delete($recursive->files, 'file');

			}

		}

	}

	/**
	* Method to send items to trash or recover item from trash.
	* 
	* @param   array  $items  List of items to be sended to trash.
	* @param   string  $type  If 'folder' or 'file' type of item.
	* @param   bool  $recover  If send to trash o recover item.  
	* 
	* @return  void
	* @since   1.0.0
	*/
	private function folder_items_trash ($items, $type, $recover) {

		switch ($type) {
			
			case 'folder':

				for ($i = 0; $i < count($items); $i++) {

					$item = new stdClass();
					$item->id = $items[$i]->id;
					
					if ($recover) {
						
						$item->trash = 0;

					} else {

						$item->trash = 2;

					}

					$db = $this->getDatabase();
					$up = $db->updateObject('#__cotton_folder', $item, 'id');
					
				}
				
				break;
			
			case 'file':
				
				for ($i = 0; $i < count($items); $i++) {

					$item = new stdClass();
					$item->id = $items[$i]->id;
					
					if ($recover) {
						
						$item->trash = 0;

					} else {

						$item->trash = 2;

					}

					$db = $this->getDatabase();
					$up = $db->updateObject('#__cotton_file', $item, 'id');
					
				}

				break;

		}

	}

	/**
	* Method to delete a item from a folder.
	* 
	* @param   int  $item_id  Id of item to be sended to trash od deleted from database.
	* @param   string  $item_type  If 'folder' or 'file' type of item.
	* @param   bool  $trash  If send to trash or delete item from database.  
	* 
	* @return  object  Status, Item Id and Item type.
	* @since   1.0.0
	*/
	public function item_delete ($item_id, $item_type, $trash) {
				
		switch ($item_type) {
			
			case 'folder':

				$this->folder_delete($item_id, $trash);
				
				if ($trash) {

					$item = new stdClass();
					$item->id = $item_id;
					
					if ($trash == 2) {

						$item->trash = 0;

					} else {

						$item->trash = 1;

					}

					$db = $this->getDatabase();
					$up = $db->updateObject('#__cotton_folder', $item, 'id');

				} else {

					$db = $this->getDatabase();
					$query = $db->getQuery(true);
					
					$query->delete($db->quoteName('#__cotton_folder'));
					$query->where('id =' . $item_id);
					
					$db->setQuery($query);
					$result = $db->execute();

				}

				break;
			
			case 'file':
				
				if ($trash) {
				
					$item = new stdClass();
					$item->id = $item_id;
					$item->trash = 1;

					$db = $this->getDatabase();
					$up = $db->updateObject('#__cotton_file', $item, 'id');

				} else {

					$db = $this->getDatabase();
					$query = $db->getQuery(true);
					
					$query->delete($db->quoteName('#__cotton_file'));
					$query->where('id =' . $item_id);
					
					$db->setQuery($query);
					$result = $db->execute();

				}
				
				break;

		}

		$data = new stdClass();
		$data->success = true;
		$data->error = '';
		$data->item_id = $item_id;
		$data->item_type = $item_type;

		return $data;

	}

	/**
	* Method to clear and delete all items in the trash from database.
	* 
	* @return  object  Status.
	* @since   1.0.0
	*/
	public function clear_trash () {
		
		$currentuser = Factory::getApplication()->getIdentity();
		$user_id = $currentuser->get("id");

		//Delete Folders
		$db = $this->getDatabase();
		$query_a = $db->getQuery(true);
		
		$query_a->delete($db->quoteName('#__cotton_folder'));
		$query_a->where('owner_id =' . $user_id);
		$query_a->where('trash != 0');
		
		$db->setQuery($query_a);
		$result = $db->execute();

		//Delete Files
		$db = $this->getDatabase();
		$query_b = $db->getQuery(true);
		
		$query_b->delete($db->quoteName('#__cotton_file'));
		$query_b->where('owner_id =' . $user_id);
		$query_b->where('trash != 0');

		$db->setQuery($query_b);
		$result = $db->execute();

		$data = new stdClass();
		$data->success = true;
		$data->error = '';

		return $data;

	}

	/**
	* Method to recover a item from trash to another folder.
	* 
	* @param   int  $item_id  Id of item to be recovered.
	* @param   string  $item_type  If 'folder' or 'file' type of item.
	* @param   string  $item_name  Name of item to be recovered for check if name exists in folder.
	* @param   int  $folder_id  Id of folder to send recovered item.
	* 
	* @return  object  Status.
	* @since   1.0.0
	*/
	public function item_recover ($item_id, $item_type, $item_name, $folder_id) {

		switch ($item_type) {

			case 'folder' :

				$item = new stdClass();
				$item->id = $item_id;
				$item->name = $this->check_item_name($folder_id, $item_name, $item_type);
				$item->parent_id = $folder_id;

				$db = $this->getDatabase();
				$up = $db->updateObject('#__cotton_folder', $item, 'id');

				$del = $this->item_delete($item_id, $item_type, 2);

				break;

			case 'file':

				$item = new stdClass();
				$item->id = $item_id;
				$item->name = $this->check_item_name($folder_id, $item_name, $item_type);
				$item->folder_id = $folder_id;
				$item->trash = 0;

				$db = $this->getDatabase();
				$up = $db->updateObject('#__cotton_file', $item, 'id');

				break;

		}

		$data = new stdClass();
		$data->success = true;
		$data->error = '';


		return $data;

	}

	/**
	* Method to delete items from database.
	* 
	* @param   array  $items  List of items to be deleted.
	* @param   string  $type  If 'folder' or 'file' type of items.
	* 
	* @return  void
	* @since   1.0.0
	*/
	public function folder_items_delete ($items, $type) {
		
		$list = [];

		for ($i = 0; $i < count($items); $i++) {

			array_push($list, $items[$i]->id);

		}

		$list = implode(',', $list);
		
		switch ($type) {
			
			case 'folder':
				
				$db = $this->getDatabase();
				$query = $db->getQuery(true);

				$query->delete($db->quoteName('#__cotton_folder'));
				$query->where('id IN ('.$list.')');

				$db->setQuery($query);
				$result = $db->execute();
				break;
			
			case 'file':
				
				$db = $this->getDatabase();
				$query = $db->getQuery(true);

				$query->delete($db->quoteName('#__cotton_file'));
				$query->where('id IN ('.$list.')');

				$db->setQuery($query);
				$result = $db->execute();
				break;

		}

	}

	/**
	* Method to upload a file item to database.
	* 
	* @param   int  $folder_id  Id of folder to insert the file.
	* @param   array  $file_uploaded  Data of file uploaded.
	* @param   string  $file_description  Description of the file uploaded.
	* @param   int  $index  Sequence index to upload multiple files.
	* @param   string  $upload_type  If is 'upload_file' or 'create_file' type.
	* 
	* @return  object  Status, Active Folder Id and index for multiple files.
	* @since   1.0.0
	*/
	public function file_upload ($folder_id, $file_uploaded, $file_description, $index, $upload_type) {

		$currentuser = Factory::getApplication()->getIdentity();
		$user_id = $currentuser->get("id");
		$limits = $this->limit_file_space();
		$data = new stdClass();

		$phpFileUploadErrors = array(	0 => Text::_('COM_COTTON_FILE_UPLOAD_ERROR_0'),
										1 => Text::_('COM_COTTON_FILE_UPLOAD_ERROR_1'),
										2 => Text::_('COM_COTTON_FILE_UPLOAD_ERROR_2'),
										3 => Text::_('COM_COTTON_FILE_UPLOAD_ERROR_3'),
										4 => Text::_('COM_COTTON_FILE_UPLOAD_ERROR_4'),
										5 => Text::_('COM_COTTON_FILE_UPLOAD_ERROR_5'), //File type error
										6 => Text::_('COM_COTTON_FILE_UPLOAD_ERROR_6'),
										7 => Text::_('COM_COTTON_FILE_UPLOAD_ERROR_7'),
										8 => Text::_('COM_COTTON_FILE_UPLOAD_ERROR_8'),
										9 => Text::_('COM_COTTON_FILE_UPLOAD_ERROR_9')
									);
		
		if ($file_uploaded != null) {
		
			if (!intval($file_uploaded['error'])) {

				if (($limits->used_space + intval($file_uploaded['size']) < $limits->limit_space) || (!$limits->limit_space)) {
				
					if ($upload_type == 'upload') {

						$fp      = fopen($file_uploaded['tmp_name'], 'r');
						$content = fread($fp, filesize($file_uploaded['tmp_name']));
						fclose($fp);

					} else {

						$content = '';

					}
					
					$db = $this->getDatabase();

					$file = new stdClass();
					$file->owner_id = $user_id;
					$file->name = $this->check_item_name($folder_id, $file_uploaded['name'], 'file');
					$file->description = $file_description;
					$file->downloads = 0;
					$file->size = $file_uploaded['size'];
					$file->mime_type = $file_uploaded['type'];
					$file->file_data = $content;
					$file->date_created = date('Y-m-d H:i:s');
					$file->date_updated = date('Y-m-d H:i:s');
					$file->folder_id = $folder_id;
					$file->allowed_users = json_encode([]);
					$file->open_link = 0;
					$file->trash = 0;
					$file->params = json_encode([]);

					$result = $db->insertObject('#__cotton_file', $file);

					$data->success = true;
					$data->error = '';

				} else {

					$data->success = false;
					$data->error = $phpFileUploadErrors[9];
		
				}

			} else {

				$data->success = false;
				$data->error = $phpFileUploadErrors[$file_uploaded['error']];

			}

		} else {

			$data->success = false;
			$data->error = $phpFileUploadErrors[5];

		}

		$data->active_folder_id = $folder_id;
		$data->index = $index;

		return $data;

	}

	/**
	* Method to check if item name exists.
	*
	* @param   int  $folder_id  Id of folder of item.
	* @param   string  $item_name  Name of file to be checked.
	* @param   string  $item_type  If 'folder' or 'file' type of item.
	* 
	* @return  string  Same item name if does not exists or item name added a number.
	* @since   1.0.0
	*/
	public function check_item_name ($folder_id, $item_name, $item_type) {

		$folder_items = $this->folder_load($folder_id);
		$i = 0;
		$ext_name = $item_name;

		switch ($item_type) {

			case 'folder':

				$items = new stdClass();
				$items->n = $folder_items->n_folders;
				$items->item = $folder_items->folders;
				break;

			case 'file':

				$items = new stdClass();
				$items->n = $folder_items->n_files;
				$items->item = $folder_items->files;
				break;

		}

		if ($items->n) {

			while ($this->check_repeat_name($items, $ext_name)) {

				$i++;

				$ext = explode('.', $item_name);

				if (count($ext) > 1) {

					$ext[count($ext)-2] .= '('.$i.')';

				} else {

					$ext[count($ext)-1] .= '('.$i.')';

				}

				$ext_name = implode('.', $ext);

			}

			if ($i == 0) {

				return $item_name;

			} else {
			
				return $ext_name;

			}

		}

		return $item_name;

	}

	/**
	* Method to check same item name in a list of items.
	*
	* @param   object  $items  List of items to check.
	* @param   string  $item_name  Name of item to be checked.
	* 
	* @return  bool  True if exists same name and False if does not.
	* @since   1.0.0
	*/
	public function check_repeat_name ($items, $item_name) {

		$i = 0;

		while ($i < $items->n) {

			if ($items->item[$i]->name == $item_name) {

				return true;

			}

			$i++;

		}

		return false;

	}

	/**
	* Method to read data file from database.
	*
	* @param   int  $file_id  Id of file to be opened.
	* @param   string  $open_type  If type of function is 'download' or 'open'.
	* 
	* @return  binary  Binary data of file with headers.
	* @since   1.0.0
	*/
	public function file_open ($file_id, $open_type) {

		$currentuser = Factory::getApplication()->getIdentity();
		$user_id = $currentuser->get("id");

		$app = Factory::getApplication();

		$result = [];

		$data = $this->file_select_open($file_id);
		
		if ($data->n) {

			$allowed = json_decode($data->file[0]->allowed_users);

			if (($data->file[0]->owner_id == $user_id) || (in_array($user_id, $allowed))) {

				$this->file_open_print($data->file[0], $open_type);

			} else {

				switch (intval($data->file[0]->open_link)) {

					case 0:
						
						$this->file_open_error(Text::_('COM_COTTON_ERROR_NOACCESS'));
						break;

					case 1:

						if ($user_id) {

							$this->file_open_print($data->file[0], $open_type);

						} else {

							$this->file_open_error(Text::_('COM_COTTON_ERROR_NOACCESS'));

						}
						

						break;

					case 2:

						$this->file_open_print($data->file[0], $open_type);
						break;

				}

			}
		
		} else {

			$this->file_open_error(Text::_('COM_COTTON_ERROR_NOFILE'));

		}

	}

	/**
	* Method to send binary data of a file with file headers.
	*
	* @param   object  $file  Data of a file to be sended.
	* @param   string  $open_type  If type of function is 'download' or 'open'.
	* 
	* @return  object  Active feed.
	* @since   1.0.0
	*/
	public function file_open_print ($file, $open_type) {

		header("Content-length: ".$file->size);
		header("Content-type: ".$file->mime_type);

		switch ($open_type) {

			case 'open':

				header('Content-Disposition: inline; filename="'.$file->name.'"');
				break;

			case 'download':

				header('Content-Disposition: attachment; filename="'.$file->name.'"');
				break;

		}

		ob_clean();
		flush();
		echo $file->file_data;

	}

	/**
	* Method to return a message error if access to open data file is denied.
	*
	* @param   string  $error  Message to be sended.
	* 
	* @return  string  Html message.
	* @since   1.0.0
	*/
	public function file_open_error ($error) {

		header('Content-Disposition: inline');
		header("Content-type: text/html");

		ob_clean();
		flush();
		echo '<div style="width: 300px;
						  height:60px;
						  background-color: #F9A541;
						  border-radius:6px; padding:20px;
						  font-family: Arial, Helvetica, sans-serif;">
					<span><h3>'.$error.'</h3><a href="'.Uri::root().'">'.Uri::root().'</a></span>
			  </div>';

	}

	/**
	* Method to select a item file information.
	*
	* @param   int  $file_id  Id of a file to be selected.
	* 
	* @return  object  Information of a file.
	* @since   1.0.0
	*/
	private function file_select ($file_id) {

		$currentuser = Factory::getApplication()->getIdentity();
		$user_id = $currentuser->get("id");

		$db = $this->getDatabase();
		$query = $db->getQuery(true);

		$query->select('a.id as id, a.owner_id as owner_id, a.name as name, a.date_created as date_created, a.date_updated as date_updated, 
						a.downloads as downloads, a.size as size, a.mime_type as mime_type, a.folder_id as folder_id, a.allowed_users as allowed_users, a.open_link as open_link, a.trash as trash, a.params as params');
		$query->from($db->quoteName('#__cotton_file', 'a'));
		$query->where($db->quoteName('a.id').'='.$file_id);
		$query->where($db->quoteName('a.owner_id').'='.$user_id);
		$query->where($db->quoteName('a.trash').'= 0');

		$db->setQuery($query);
		$db->execute();

		$data = new stdClass();
		$data->n = $db->getNumRows();
		$data->file = $db->loadObjectList();

		return $data;

	}

	/**
	* Method to select a item file data.
	*
	* @param   int  $file_id  Id of a file to be opened.
	* 
	* @return  object  Data of a file.
	* @since   1.0.0
	*/
	private function file_select_open ($file_id) {

		$db = $this->getDatabase();
		$query = $db->getQuery(true);

		$query->select('a.id as id, a.owner_id as owner_id, a.name as name, a.date_created as date_created, a.date_updated as date_updated, 
						a.downloads as downloads, a.size as size, a.mime_type as mime_type, a.file_data as file_data, a.folder_id as folder_id, a.allowed_users as allowed_users, a.open_link as open_link, a.trash as trash, a.params as params');
		$query->from($db->quoteName('#__cotton_file', 'a'));
		$query->where($db->quoteName('a.id').'='.$file_id);
		$query->where($db->quoteName('a.trash').'= 0');

		$db->setQuery($query);
		$db->execute();

		$data = new stdClass();
		$data->n = $db->getNumRows();
		$data->file = $db->loadObjectList();

		return $data;

	}

	/**
	* Method to save text file data edited by CodeMirror in database.
	* 
	* @param   int  $file_id  Id of file to be saved.
	* @param   text  $file_data  Text of file to be saved.
	* @param   int  $file_size  Size of the text in bytes.
	* 
	* @return  object  Status, Id of the file and saved data.
	* @since   1.0.0
	*/
	public function file_save ($file_id, $file_saved) {

		$limits = $this->limit_file_space();
		$data = new stdClass();

		if (!intval($file_saved['error'])) {

			if (($limits->used_space + intval($file_saved['size']) < $limits->limit_space) || (!$limits->limit_space)) {
			

				$fp      = fopen($file_saved['tmp_name'], 'r');
				$content = fread($fp, filesize($file_saved['tmp_name']));
				fclose($fp);

				$file = new stdClass();
				$file->id = $file_id;
				$file->file_data = $content;
				$file->size = $file_saved['size'];
				$file->date_updated = date('Y-m-d H:i:s');

				$db = $this->getDatabase();

				$up = $db->updateObject('#__cotton_file', $file, 'id');

				$data->success = true;
				$data->error = '';
				$data->file_id = $file_id;
				$data->file_data = $content;

			} else {

				$data->success = false;
				$data->error = Text::_('COM_COTTON_FILE_UPLOAD_ERROR_9');
				$data->file_id = $file_id;

			}

		} else {

			$data->success = false;
			$data->error = Text::_('COM_COTTON_FILE_UPLOAD_ERROR_' . $file_saved['error']);
			$data->file_id = $file_id;

		}

		return $data;

	}

	/**
	* Method to update information of a file item.
	* 
	* @param   int  $file_id  Id of file to be updated.
	* @param   string  $file_name  Name of file to be updated.
	* @param   string  $file_description  Description of file to be updated.
	* @param   int  $file_open_link  Type of share of file to be updated [0: "No shared", 1: "Shared with Registered Users", 2: "Shared with anyone"].
	* @param   array  $file_allowed_users  List of user IDs with allowed access to this file.
	* 
	* @return  object  File information updated.
	* @since   1.0.0
	*/
	public function file_update ($file_id, $file_name, $file_description, $file_open_link, $file_allowed_users) {

		$file_allowed_users = json_decode($file_allowed_users);

		$allowed = [];

		for ($i = 0; $i < count($file_allowed_users); $i++) {

			if (intval($file_allowed_users[$i])) {

				array_push($allowed, intval($file_allowed_users[$i]));

			}

		}
		
		$file = new stdClass();
		$file->id = $file_id;
		$file->name = $file_name;
		$file->description = $file_description;
		$file->date_updated = date('Y-m-d H:i:s');
		$file->open_link = $file_open_link;
		$file->allowed_users = json_encode($allowed);

		$db = $this->getDatabase();

		$up = $db->updateObject('#__cotton_file', $file, 'id');

		return $file;

	}

	/**
	* Method to delete a file item.
	* 
	* @param   int  $file_id  Id of file to be deleted.
	* @param   int  $folder_id  Id of folder to be reloaded after to delete.
	* @param   bool  $trash  If item file is going to the trash or to be deleted from database.
	* 
	* @return  object  Data folder reloaded after item file to be deleted.
	* @since   1.0.0
	*/
	public function file_delete ($file_id, $folder_id, $trash) {

		$del = $this->file_select($file_id);
		
		if ($del->n) {
		
			for ($p = 0; $p < $del->n; $p++) {

				if ($trash) {
					
					$this->trash([$data->file[$p]], 'file');

				} else {

					$this->folder_items_delete([$data->file[$p]], 'file');

				}

			}

			$data = $this->folder_load($folder_id);
			$data->error = false;

			return $data;

		} else {

			$data = new stdClass();
			$data->error = true;

			return $data;

		}

	}

/**
	* Method to load all items in the trash.
	* 
	* @return  object  List of folder and files in the trash, list of folder tree and space limits configurations.
	* @since   1.0.0
	*/
	public function load_trash () {

		$currentuser = Factory::getApplication()->getIdentity();
		$user_id = $currentuser->get("id");
		
		$data = new stdClass();
		
		$db = $this->getDatabase();
		$query = $db->getQuery(true);

		$query->select('a.id as id, a.owner_id as owner_id, a.name as name, a.description as description, a.date_created as date_created, a.date_updated as date_updated, 
						a.parent_id as parent_id, a.allowed_users as allowed_users, a.open_link as open_link, a.trash as trash, a.params as params');
		$query->from($db->quoteName('#__cotton_folder', 'a'));
		$query->where($db->quoteName('a.owner_id').'='.$user_id);
		$query->where($db->quoteName('a.trash').'= 1');
		$query->group('a.id');
		$query->select('COUNT(DISTINCT b.id) as n_folders');
		$query->join('LEFT', $db->quoteName('#__cotton_folder', 'b') . ' ON b.parent_id = a.id AND b.owner_id = ' . $user_id);
		$query->select('COUNT(DISTINCT c.id) as n_files');
		$query->join('LEFT', $db->quoteName('#__cotton_file', 'c') . ' ON c.folder_id = a.id AND c.owner_id = ' . $user_id);
		$query->order($db->quoteName('name'));

		$db->setQuery($query);
		$db->execute();

		$data->n_folders_trash = $db->getNumRows();
		$data->folders_trash = $db->loadObjectList();
		
		$files_trash = $this->folder_load_files_trash();

		$data->n_files_trash = $files_trash->n;
		$data->files_trash = $files_trash->files;

		$folder_list = $this->folder_load_list();

		$data->n_list = $folder_list->n;
		$data->list = $folder_list->list;

		$data->limits = $this->limit_file_space();
		
		$data->path = '/Trash';

		return $data;

	}

	/**
	* Method to select items files in the trash.
	* 
	* @return  object  List of items files in the trash.
	* @since   1.0.0
	*/
	private function folder_load_files_trash () {

		$currentuser = Factory::getApplication()->getIdentity();
		$user_id = $currentuser->get("id");

		$db = $this->getDatabase();
		$query = $db->getQuery(true);

		$query->select('a.id as id, a.owner_id as owner_id, a.name as name, a.description as description, a.date_created as date_created, a.date_updated as date_updated, 
						a.downloads as downloads, a.size as size, a.mime_type as mime_type, a.folder_id as folder_id, a.allowed_users as allowed_users, a.open_link as open_link, a.trash as trash, a.params as params');
		$query->from($db->quoteName('#__cotton_file', 'a'));
		$query->where($db->quoteName('a.owner_id').'='.$user_id);
		$query->where($db->quoteName('a.trash').'= 1');
		$query->order($db->quoteName('name'));

		$db->setQuery($query);
		$db->execute();

		$data = new stdClass();
		$data->n = $db->getNumRows();
		$data->files = $db->loadObjectList();

		return $data;

	}

	/**
	* Method to space limits configuration of Cotton Cloud component.
	* 
	* @return  object  Limit Space per User, Used Space, Trash Used Space and Upload Max File Size.
	* @since   1.0.0
	*/
	public function limit_file_space () {

		$app = Factory::getApplication();
		$config = $app->getParams('com_cotton');
		$currentuser = Factory::getApplication()->getIdentity();
		$user_id = $currentuser->get("id");
		
		$data = new stdClass();
		
		if (intval($config->get('cotton_limit_space'))) {

			$data->limit_space = intval($config->get('cotton_limit_space'));

		} else {

			$data->limit_space = 0;

		}

		$db = $this->getDatabase();
		$query = $db->getQuery(true);

		$query->select('SUM(a.size) as used_space');
		$query->from($db->quoteName('#__cotton_file', 'a'));
		$query->where($db->quoteName('a.owner_id').'='.$user_id);

		$db->setQuery($query);
		$db->execute();

		$n = $db->getNumRows();
		$used_space = $db->loadObjectList();

		if ($n) {

			$data->used_space = intval($used_space[0]->used_space);

		} else {

			$data->used_space = 0;

		}

		$db = $this->getDatabase();
		$query_b = $db->getQuery(true);

		$query_b->select('SUM(a.size) as trash_used_space');
		$query_b->from($db->quoteName('#__cotton_file', 'a'));
		$query_b->where($db->quoteName('a.owner_id').'='.$user_id);
		$query_b->where($db->quoteName('a.trash').'= 1');

		$db->setQuery($query_b);
		$db->execute();

		$n = $db->getNumRows();
		$trash_used_space = $db->loadObjectList();

		if ($n) {

			$data->trash_used_space = intval($trash_used_space[0]->trash_used_space);

		} else {

			$data->trash_used_space = 0;

		}

		$val = trim(ini_get('upload_max_filesize'));
		$last = strtolower($val[strlen($val)-1]);
		$result = '';

		for ($i = 0; $i < strlen($val); $i++) {

			$result .= $val[$i];

		}

		$result = intval($result);		
		
		switch ($last) {
			case 'g':
				$result *= 1024;
			case 'm':
				$result *= 1024;
			case 'k':
				$result *= 1024;
		}

		$data->max_filesize = $result;

		if ($data->max_filesize > $this->max_filesize_db) {

			$data->max_filesize = $this->max_filesize_db;

		}

	return $data;

	}

	/**
     * Method to send file data to be opened in CodeMirror editor.
     *
     * @param   int  $file_id  Id of the file to be opened.
     * @param   string  $file_ext  Extension of the file name to check file mime type.
     *
     * @return  object  File data to be opened in CodeMirror editor.
     *
     * @since   1.6
     */
    public function open_editor($file_id, $file_ext) {
		
		$app = Factory::getApplication();

        // Codemirror or Editor None should be enabled
        $db    = $this->getDatabase();
        $query = $db->getQuery(true);
		
		$query->select('COUNT(*) as state');
		$query->from('#__extensions as a');
		$query->where('(a.name =' . $db->quote('plg_editors_codemirror') . ' AND a.enabled = 1)');
        
		$db->setQuery($query);
        $db->execute();

		$n = $db->getNumRows();
		$state = $db->loadObjectList();

        $data = new stdClass();
		
		if ($state[0]->state < 1) {

            $data->file_id = $file_id;
			$data->file_ext = $file_ext;
			$data->success = false;
			$data->error = Text::_('COM_COTTON_ERROR_EDITOR_DISABLED');

        } else {
		
			$file = $this->file_select_open($file_id);

			$data->file_id = $file_id;
			$data->file_ext = $file_ext;
			$data->file_data = $file->file[0]->file_data;
			$data->success = true;
			$data->error = '';

		}

        return $data;

    }

	/**
	* Method to convert a string of files formats extensions in an array to be checked.
	* 
	* @return  object  Active feed.
	* @since   1.0.0
	*/
	private function array_formats ($list) {

		$formats = explode(',', $list);

		for ($i = 0; $i < count($formats); $i++) {

			$formats[$i] = trim($formats[$i]);
			$formats[$i] = strtolower($formats[$i]);

		}

		return $formats;

	}
	
	/**
	* Method to get Joomla database driver.
	* 
	* @return  object  DatabaseDriver.
	* @since   1.0.0
	*/
	private function getDatabase () {

		return Factory::getContainer()->get('DatabaseDriver');

	}

}
