<?php
/**
 * @package Tabaoca.Component.Cotton.Site
 * @subpackage com_cotton
 * @copyright (C) 2024 Jonatas C. Ferreira
 * @license GNU/AGPL v3 https://www.gnu.org/licenses/agpl-3.0.html
 */

namespace Tabaoca\Component\Cotton\Site\Controller;

\defined('_JEXEC') or die;

use stdClass;
use Joomla\CMS\MVC\Controller\BaseController;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Session\Session;
use Joomla\CMS\Response\JsonResponse;

/**
 * Controller of Cotton Cloud System component
 *
 * @package     Tabaoca.Component.Cotton.Site
 * @subpackage  com_cotton
 * @since       1.0.0
 */
class CottonController extends BaseController {

	/**
	* Method to start the system with current configuration.
	* 
	* @return  object  Response JSON encoded object to XHR call.
	* @since   1.0.0
	*/
	public function start () {

		try {

			if (!Session::checkToken()) {

				echo new JsonResponse(null, Text::_('JINVALID_TOKEN'), true);

			} else {

				$input = Factory::getApplication()->input;

				$model = $this->getModel();

				$record = $model->start();

				echo new JsonResponse($record);

			}

		} catch (Exception $e) {

			echo new JsonResponse($e);

		}

	}

	/**
	* Method to create a new folder.
	* 
	* @return  object  Response JSON encoded object to XHR call.
	* @since   1.0.0
	*/
	public function folder_create () {

		try {

			if (!Session::checkToken()) {

				echo new JsonResponse(null, Text::_('JINVALID_TOKEN'), true);

			} else {

				$input = Factory::getApplication()->input;

				$parent_id = $input->get('parent_id', 0, 'INT');
				$folder_name = $input->get('folder_name', '', 'STRING');
				$folder_description = $input->get('folder_description', '', 'STRING');

				$model = $this->getModel();

				$record = $model->folder_create($parent_id, $folder_name, $folder_description);

				echo new JsonResponse($record);

			}

		} catch (Exception $e) {

			echo new JsonResponse($e);

		}

	}

	/**
	* Method to load folder content information.
	* 
	* @return  object  Response JSON encoded object to XHR call.
	* @since   1.0.0
	*/
	public function folder_load () {

		try {

			if (!Session::checkToken()) {

				echo new JsonResponse(null, Text::_('JINVALID_TOKEN'), true);

			} else {

				$input = Factory::getApplication()->input;

				$folder_id = $input->get('folder_id', 0, 'INT');

				$model = $this->getModel();

				$record = $model->folder_load($folder_id);

				echo new JsonResponse($record);

			}

		} catch (Exception $e) {

			echo new JsonResponse($e);

		}

	}

	/**
	* Method to update folder data.
	* 
	* @return  object  Response JSON encoded object to XHR call.
	* @since   1.0.0
	*/
	public function folder_update () {

		try {

			if (!Session::checkToken()) {

				echo new JsonResponse(null, Text::_('JINVALID_TOKEN'), true);

			} else {

				$input = Factory::getApplication()->input;

				$folder_id = $input->get('folder_id', 0, 'INT');
				$folder_name = $input->get('folder_name', '', 'STRING');
				$folder_description = $input->get('folder_description', '', 'STRING');

				$model = $this->getModel();

				$record = $model->folder_update($folder_id, $folder_name, $folder_description);

				echo new JsonResponse($record);

			}

		} catch (Exception $e) {

			echo new JsonResponse($e);

		}

	}

	/**
	* Method to delete a folder or a file.
	* 
	* @return  object  Response JSON encoded object to XHR call.
	* @since   1.0.0
	*/
	public function item_delete () {

		try {

			if (!Session::checkToken()) {

				echo new JsonResponse(null, Text::_('JINVALID_TOKEN'), true);

			} else {

				$input = Factory::getApplication()->input;

				$item_id = $input->get('item_id', 0, 'INT');
				$item_type = $input->get('item_type', '', 'STRING');
				$trash = $input->get('trash', 0, 'INT');

				$model = $this->getModel();

				$record = $model->item_delete($item_id, $item_type, $trash);

				echo new JsonResponse($record);

			}

		} catch (Exception $e) {

			echo new JsonResponse($e);

		}

	}

	/**
	* Method to clear all items in the Trash.
	* 
	* @return  object  Response JSON encoded object to XHR call.
	* @since   1.0.0
	*/
	public function clear_trash () {

		try {

			if (!Session::checkToken()) {

				echo new JsonResponse(null, Text::_('JINVALID_TOKEN'), true);

			} else {

				$input = Factory::getApplication()->input;

				$model = $this->getModel();

				$record = $model->clear_trash();

				echo new JsonResponse($record);

			}

		} catch (Exception $e) {

			echo new JsonResponse($e);

		}

	}

	/**
	* Method to recover an item from Trash to another folder.
	* 
	* @return  object  Response JSON encoded object to XHR call.
	* @since   1.0.0
	*/
	public function item_recover () {

		try {

			if (!Session::checkToken()) {

				echo new JsonResponse(null, Text::_('JINVALID_TOKEN'), true);

			} else {

				$input = Factory::getApplication()->input;

				$item_id = $input->get('item_id', 0, 'INT');
				$item_type = $input->get('item_type', '', 'STRING');
				$item_name = $input->get('item_name', '', 'STRING');
				$folder_id = $input->get('folder_id', 0, 'INT');

				$model = $this->getModel();

				$record = $model->item_recover($item_id, $item_type, $item_name, $folder_id);

				echo new JsonResponse($record);

			}

		} catch (Exception $e) {

			echo new JsonResponse($e);

		}

	}

	/**
	* Method to create a new text file in the database.
	* 
	* @return  object  Response JSON encoded object to XHR call.
	* @since   1.0.0
	*/
	public function file_create () {

		try {

			if (!Session::checkToken()) {

				echo new JsonResponse(null, Text::_('JINVALID_TOKEN'), true);

			} else {

				$input = Factory::getApplication()->input;

				$folder_id = $input->get('folder_id', 0, 'INT');
				$file = $input->files->get('file', null, 'RAW');
				$file_description = $input->get('file_description', '', 'STRING');
				
				$file_ext = explode('.', $file['name']);

				switch ($file_ext[count($file_ext) - 1]) {

					case 'htm':
					case 'html':

						$file['type'] = 'text/html';
						break;

					case 'xhtml':

						$file['type'] = 'application/xhtml+xml';
						break;

					case 'php':

						$file['type'] = 'application/x-httpd-php';
						break;

					case 'js':
					case 'mjs':

						$file['type'] = 'text/javascript';
						break;

					case 'xml':

						$file['type'] = 'application/xml';
						break;

					case 'json':

						$file['type'] = 'application/json';
						break;

					case 'jsonld':

						$file['type'] = 'application/ld+json';
						break;

					case 'csv':

						$file['type'] = 'text/csv';
						break;

					case 'css':
					case 'scss':
					case 'less':
					case 'sass':

						$file['type'] = 'text/css';
						break;

					case 'xul':

						$file['type'] = 'application/vnd.mozilla.xul+xml';
						break;

					default:

						$file['type'] = 'text/plain';
						break;

				}

				$model = $this->getModel();

				$record = $model->file_upload($folder_id, $file, $file_description, 0, 'create');

				echo new JsonResponse($record);

			}

		} catch (Exception $e) {

			echo new JsonResponse($e);

		}

	}

	/**
	* Method to upload a file to database.
	* 
	* @return  object  Response JSON encoded object to XHR call.
	* @since   1.0.0
	*/
	public function file_upload () {

		try {

			if (!Session::checkToken()) {

				echo new JsonResponse(null, Text::_('JINVALID_TOKEN'), true);

			} else {

				$input = Factory::getApplication()->input;

				$folder_id = $input->get('folder_id', 0, 'INT');
				$file = $input->files->get('file', null, 'RAW');
				$index = $input->get('index', 0, 'INT');

				$model = $this->getModel();

				$record = $model->file_upload($folder_id, $file, '', $index, 'upload');

				echo new JsonResponse($record);

			}

		} catch (Exception $e) {

			echo new JsonResponse($e);

		}

	}

	/**
	* Method to download a file from database like attachment.
	* 
	* @return  object  Response JSON encoded object to XHR call.
	* @since   1.0.0
	*/
	public function download () {

		try {

			if (!Session::checkToken()) {

				echo new JsonResponse(null, Text::_('JINVALID_TOKEN'), true);

			} else {

				$input = Factory::getApplication()->input;

				$file_id = $input->get('file_id', 0, 'INT');

				$model = $this->getModel();

				$record = $model->file_open($file_id, 'download');

			}

		} catch (Exception $e) {

			echo new JsonResponse($e);

		}

	}

	/**
	* Method to get data from a file.
	* 
	* @return  object  Response JSON encoded object to XHR call.
	* @since   1.0.0
	*/
	public function open () {

		//Do not check form token.

		try {

			$input = Factory::getApplication()->input;

			$file_id = $input->get('file_id', 0, 'INT');

			$model = $this->getModel();

			$record = $model->file_open($file_id, 'open');


		} catch (Exception $e) {

			echo new JsonResponse($e);

		}

	}

	/**
	* Method to save a text file data.
	* 
	* @return  object  Response JSON encoded object to XHR call.
	* @since   1.0.0
	*/
	public function file_save () {

		try {

			if (!Session::checkToken()) {

				echo new JsonResponse(null, Text::_('JINVALID_TOKEN'), true);

			} else {

				$input = Factory::getApplication()->input;

				$file_id = $input->get('file_id', 0, 'INT');
				$file_saved = $input->files->get('file', null, 'RAW');

				$model = $this->getModel();

				$record = $model->file_save($file_id, $file_saved);

				echo new JsonResponse($record);

			}

		} catch (Exception $e) {

			echo new JsonResponse($e);

		}

	}

	/**
	* Method to update data information of a file.
	* 
	* @return  object  Response JSON encoded object to XHR call.
	* @since   1.0.0
	*/
	public function file_update () {

		try {

			if (!Session::checkToken()) {

				echo new JsonResponse(null, Text::_('JINVALID_TOKEN'), true);

			} else {

				$input = Factory::getApplication()->input;

				$file_id = $input->get('file_id', 0, 'INT');
				$file_name = $input->get('file_name', '', 'STRING');
				$file_description = $input->get('file_description', '', 'STRING');
				$file_open_link = $input->get('file_open_link', 0, 'INT');
				$file_allowed_users = $input->get('file_allowed_users', '', 'STRING');

				$model = $this->getModel();

				$record = $model->file_update($file_id, $file_name, $file_description, $file_open_link, $file_allowed_users);

				echo new JsonResponse($record);

			}

		} catch (Exception $e) {

			echo new JsonResponse($e);

		}

	}

	/**
	* Method to load all items in the trash.
	* 
	* @return  object  Response JSON encoded object to XHR call.
	* @since   1.0.0
	*/
	public function load_trash () {

		try {

			if (!Session::checkToken()) {

				echo new JsonResponse(null, Text::_('JINVALID_TOKEN'), true);

			} else {

				$input = Factory::getApplication()->input;

				$model = $this->getModel();

				$record = $model->load_trash();

				echo new JsonResponse($record);

			}

		} catch (Exception $e) {

			echo new JsonResponse($e);

		}

	}

	/**
	* Method to get data text from a file to open in CodeMirror Editor.
	* 
	* @return  object  Response JSON encoded object to XHR call.
	* @since   1.0.0
	*/
	public function open_editor () {

		try {

			if (!Session::checkToken()) {

				echo new JsonResponse(null, Text::_('JINVALID_TOKEN'), true);

			} else {

				$input = Factory::getApplication()->input;

				$model = $this->getModel();

				$file_id = $input->get('file_id', 0, 'INT');
				$file_ext = $input->get('file_ext', '', 'STRING');

				$editor = $model->open_editor($file_id, $file_ext);

				echo new JsonResponse($editor);

			}

		} catch (Exception $e) {

			echo new JsonResponse($e);

		}

	}

}
